/************************************************************************
 * @file: gst_dlt_internals.h
 *
 * @version: 0.1
 *
 * @description: This header file contains all the typedef, internal struct
 * Mapping of GST-DLT log levels. Macros for wait times and max threads.
 *
 * @authors: Athreya Manasa Balaraj, ManasaBalaraj.Athreya@in.bosch.com 2017
 *           Devi Laxmi, Laxmi.Devi@in.bosch.com 2017
 *
 * @copyright (c) 2015 Advanced Driver Information Technology.
 * Copyright of Advanced Driver Information Technology, Bosch, and DENSO.
 * All rights reserved.
 *
 ***********************************************************************/

#ifndef __internals_gst_dlt_H__
#define __internals_gst_dlt_H__

/*! @file gst_dlt_internals.h
 *  Describes the functions that are used internally within GDA interface.
 */

#ifdef __cplusplus
extern "C"
{
#endif

#include <dlt/dlt.h>
#include <gst/gst.h>
#include <stdio.h>
#include <stdlib.h>
#include <stdbool.h>

#include "gst_dlt_adapter.h"

/** @brief Macro for GDA Default Context ID and description */
#define GDA_DEFAULT_CTXID "GDAC"
#define GDA_DEFAULT_CTXID_DES "GDA Default DLT context"

/** @brief Macro for max number of threads per context */
#define MAX_THREADS 64

/** @brief Enum to map DLT Log levels to Gstreamer Log levels.
  * DLT_LOG_MAX is the maximum log level in DLT
  */
const GstDebugLevel dlt2gst[DLT_LOG_MAX]=
{
    [DLT_LOG_OFF]     = GST_LEVEL_NONE,
    [DLT_LOG_FATAL]   = GST_LEVEL_ERROR,
    [DLT_LOG_ERROR]   = GST_LEVEL_ERROR,
    [DLT_LOG_WARN]    = GST_LEVEL_WARNING,
    [DLT_LOG_INFO]    = GST_LEVEL_INFO,
    [DLT_LOG_DEBUG]   = GST_LEVEL_DEBUG,
    [DLT_LOG_VERBOSE] = GST_LEVEL_LOG,
};

/** @brief Enum to map Gstreamer Log levels to DLT Log levels.
  * GST_LEVEL_COUNT is the maximum log level in Gstreamer
  */
const DltLogLevelType gst2dlt[GST_LEVEL_COUNT]={
    [GST_LEVEL_NONE]    = DLT_LOG_OFF,
    [GST_LEVEL_ERROR]   = DLT_LOG_ERROR,
    [GST_LEVEL_WARNING] = DLT_LOG_WARN,
    [GST_LEVEL_INFO]    = DLT_LOG_INFO,
    [GST_LEVEL_DEBUG]   = DLT_LOG_DEBUG,
    [GST_LEVEL_LOG]     = DLT_LOG_VERBOSE,
    [GST_LEVEL_FIXME]   = DLT_LOG_VERBOSE,
    [GST_LEVEL_TRACE]   = DLT_LOG_VERBOSE,
    [GST_LEVEL_MEMDUMP] = DLT_LOG_VERBOSE,
};

/** @brief Structure to maintain list of category and level provided by Injection Callback */
typedef struct gda_log_list
{
    char*                category;
    uint32_t             level;
    struct gda_log_list* next;
}gda_log_list_t;

/** @brief Structure to maintain list of Gstreamer threads per context */
typedef struct thread_info
{
    GThread*    gda_thread_id_arr[MAX_THREADS];
    uint32_t    gda_thread_count;
}thread_info_t;

/** @brief Structure to maintain list of Gstreamer pipelines created */
typedef struct gda_ctx_ll
{
    GDA_CTX*            ctx;
    gda_log_list_t*     gda_log_list;
    thread_info_t*      thread_info;
    struct gda_ctx_ll*  next_ctx;
} gda_ctx_ll_t;

/** This function will be called to check if the given string is a number.
  * @param[IN] str  Specifies any string which is to be checked
  * @return 0       If String contains characters
  * @return 1       If String contains numbers
  */
static bool gda_check_numbers (const char * str);

/** This function will be called to verify if Injection callback category matches with any
    Gstreamer log category.
  * @param[IN] first   Specifies string-1
  * @param[IN] second  Specifies string-2
  * @return 0          If first string and second string doesn't match
  * @return 1          If first string and second string match
  */
static bool gda_match (char * first, const char * second);

/** This is the callback that is invoked when any Injection message is invoked by DLT viewer.
  * @param[IN] service_id    Specifies the ServiceID for which Injection Message is
                             configured from DLT viewer
  * @param[IN] inj_data      Contains Injection Message(Text/Binary data)
  * @param[IN] length        Gives length of the Injection Message
  * @param[IN] data          User data
  * @return -1               If injection fails to allocate memory
  */
static int gda_filter_category_callback(uint32_t service_id, void * inj_data, uint32_t length, void * data);

/** This is the function to compare the GStreamer threadIDs against the registered ThreadIDs
  * @param[IN] thread_info   Thread info struct containing array for thread IDs and count
  * @param[IN] thread_id     Thread ID of the GStreamer thread
  * @return 1                True if the thread_id is found in the array of registered
  * @return 0                Flase if the thread_id is not found in the array of registered
  */
static int gda_compare_thread_id (thread_info_t * thread_info, GThread *thread_id);

/** This is the function to frame the Gstreamer log message
  * @param[IN] category     Specifies gstreamer category of the log message
  * @param[IN] file         Specifies gstreamer file from which this log Message is generated
  * @param[IN] function     Specifies gstreamer function which has the log Message
  * @param[IN] line         Line number of the log Message
  * @param[IN] object       Gives the object of the log Message
  * @param[IN] message      Log Message
  * @return    msg          The Gstreamer log message
  */
static gchar *
gda_frame_debug_log_msg (GstDebugCategory * category, const gchar * file,
        const gchar * function, gint line, GObject * object, GstDebugMessage * message);

/** This is the logging function provided to GStreamer gst_debug_add_log_function() call.
  * @param[IN] category     Specifies gstreamer category of the Log message
  * @param[IN] level        Specifies gstreamer level of the Log message
  * @param[IN] file         Specifies gstreamer file from which this Log Message is generated
  * @param[IN] function     Specifies gstreamer function which has the Log Message
  * @param[IN] line         Line number of the Log Message
  * @param[IN] object       Gives the object of the Log Message
  * @param[IN] message      Log Message
  * @param[IN] unused       Any pointer
  */
static void
gda_get_debug_log (GstDebugCategory * category, GstDebugLevel level,
    const gchar * file, const gchar * function, gint line,
    GObject * object, GstDebugMessage * message, gpointer unused);

/** This is the callback that is invoked, if dlt viewer changes the log level of the context.
  * @param[IN] context_id    Specifies the context id of the context whose log level is
                             changed in dlt viewer
  * @param[IN] log_level     Specifies log level set for the context
  * @param[IN] trace_status  Specifies trace status(ON/OFF)
  */
static void
gda_user_set_log_level_callback (char context_id[DLT_ID_SIZE], uint8_t log_level, uint8_t trace_status);

#ifdef __cplusplus
}
#endif

#endif /* __internals_gst_dlt_H__ */
